# bot/utils/webhook_manager.py
import json
import os
from pathlib import Path
from aiogram import Bot
from dotenv import load_dotenv
from bot.utils.error_reporter import report_error

CONFIG_PATH = Path("webhook_config.json")

load_dotenv()
ADMINS = [int(x) for x in os.getenv("ADMINS", "").split(",") if x]
ADMIN_ID = ADMINS[0] if ADMINS else None


def load_config():
    if CONFIG_PATH.exists():
        return json.loads(CONFIG_PATH.read_text())
    return {"enabled": False, "domain": "", "path": "/webhook", "port": 8080}


def save_config(data: dict):
    CONFIG_PATH.write_text(json.dumps(data, indent=2, ensure_ascii=False))


def get_full_url() -> str:
    """برمی‌گرداند URL کامل وبهوک در صورت فعال بودن"""
    cfg = load_config()
    if not cfg.get("domain"):
        return ""
    return f"{cfg['domain'].rstrip('/')}{cfg['path']}"


async def activate_webhook(bot: Bot):
    cfg = load_config()
    if not cfg["domain"]:
        return False, "❌ دامنه تنظیم نشده است."
    try:
        full_url = get_full_url()
        await bot.set_webhook(full_url)
        cfg["enabled"] = True
        save_config(cfg)
        return True, f"✅ وبهوک فعال شد:\n<code>{full_url}</code>"
    except Exception as e:
        if ADMIN_ID:
            await report_error(bot, ADMIN_ID, "فعال‌سازی وبهوک", e)
        return False, f"❌ خطا در فعال‌سازی وبهوک:\n<code>{str(e)}</code>"


async def deactivate_webhook(bot: Bot):
    try:
        await bot.delete_webhook()
        cfg = load_config()
        cfg["enabled"] = False
        save_config(cfg)
        return True, "❌ وبهوک غیرفعال شد."
    except Exception as e:
        if ADMIN_ID:
            await report_error(bot, ADMIN_ID, "غیرفعال‌سازی وبهوک", e)
        return False, f"❌ خطا در غیرفعال‌سازی وبهوک:\n<code>{str(e)}</code>"
